<?php
/**
 * MetinPaylaş - Admin Kullanıcı Yönetimi
 */

require_once '../includes/config.php';
require_once '../includes/functions.php';

// Admin girişi kontrolü
if (!isset($_SESSION['admin_giris']) || $_SESSION['admin_giris'] !== true) {
    header('Location: index.php');
    exit;
}

// Ayarları getir
$ayarlar = ayarlariGetir();
$varsayilan_tema = isset($ayarlar['tema']) ? $ayarlar['tema'] : 'light';

$db = baglan();

// Kullanıcı işlemleri
$mesaj = '';
$mesaj_turu = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['kullanici_ekle'])) {
        // Yeni kullanıcı ekleme
        $kullanici_adi = trim($_POST['kullanici_adi']);
        $email = trim($_POST['email']);
        $sifre = $_POST['sifre'];
        $ad_soyad = trim($_POST['ad_soyad']);
        $yetki = $_POST['yetki'];
        
        // Validation
        if (empty($kullanici_adi) || empty($email) || empty($sifre)) {
            $mesaj = 'Kullanıcı adı, email ve şifre gereklidir.';
            $mesaj_turu = 'danger';
        } else {
            // Kullanıcı adı ve email kontrolü
            $check = $db->prepare("SELECT COUNT(*) as count FROM kullanicilar WHERE kullanici_adi = ? OR email = ?");
            $check->execute([$kullanici_adi, $email]);
            
            if ($check->fetch()['count'] > 0) {
                $mesaj = 'Bu kullanıcı adı veya email zaten kullanımda.';
                $mesaj_turu = 'danger';
            } else {
                $sifre_hash = password_hash($sifre, PASSWORD_DEFAULT);
                $insert = $db->prepare("INSERT INTO kullanicilar (kullanici_adi, email, sifre, ad_soyad, yetki) VALUES (?, ?, ?, ?, ?)");
                
                if ($insert->execute([$kullanici_adi, $email, $sifre_hash, $ad_soyad, $yetki])) {
                    $mesaj = 'Kullanıcı başarıyla eklendi.';
                    $mesaj_turu = 'success';
                } else {
                    $mesaj = 'Kullanıcı eklenirken bir hata oluştu.';
                    $mesaj_turu = 'danger';
                }
            }
        }
    } elseif (isset($_POST['kullanici_duzenle'])) {
        // Kullanıcı düzenleme
        $id = $_POST['kullanici_id'];
        $kullanici_adi = trim($_POST['kullanici_adi']);
        $email = trim($_POST['email']);
        $ad_soyad = trim($_POST['ad_soyad']);
        $yetki = $_POST['yetki'];
        $durum = isset($_POST['durum']) ? 1 : 0;
        
        $update = $db->prepare("UPDATE kullanicilar SET kullanici_adi = ?, email = ?, ad_soyad = ?, yetki = ?, durum = ? WHERE id = ?");
        
        if ($update->execute([$kullanici_adi, $email, $ad_soyad, $yetki, $durum, $id])) {
            // Şifre değiştirilecek mi?
            if (!empty($_POST['yeni_sifre'])) {
                $yeni_sifre_hash = password_hash($_POST['yeni_sifre'], PASSWORD_DEFAULT);
                $update_sifre = $db->prepare("UPDATE kullanicilar SET sifre = ? WHERE id = ?");
                $update_sifre->execute([$yeni_sifre_hash, $id]);
            }
            
            $mesaj = 'Kullanıcı başarıyla güncellendi.';
            $mesaj_turu = 'success';
        } else {
            $mesaj = 'Kullanıcı güncellenirken bir hata oluştu.';
            $mesaj_turu = 'danger';
        }
    } elseif (isset($_POST['kullanici_sil'])) {
        // Kullanıcı silme
        $id = $_POST['kullanici_id'];
        
        // Kendi hesabını silmeye çalışıyor mu?
        if ($id == $_SESSION['admin_id']) {
            $mesaj = 'Kendi hesabınızı silemezsiniz.';
            $mesaj_turu = 'danger';
        } else {
            $delete = $db->prepare("DELETE FROM kullanicilar WHERE id = ?");
            if ($delete->execute([$id])) {
                $mesaj = 'Kullanıcı başarıyla silindi.';
                $mesaj_turu = 'success';
            } else {
                $mesaj = 'Kullanıcı silinirken bir hata oluştu.';
                $mesaj_turu = 'danger';
            }
        }
    }
}

// Kullanıcıları getir
$kullanicilar = $db->query("SELECT * FROM kullanicilar ORDER BY kayit_tarihi DESC")->fetchAll();

// İstatistikler
$toplam_kullanici = count($kullanicilar);
$aktif_kullanici = count(array_filter($kullanicilar, fn($k) => $k['durum'] == 1));
$admin_sayisi = count(array_filter($kullanicilar, fn($k) => $k['yetki'] == 'admin'));
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kullanıcılar - Admin Panel</title>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/admin.css">
</head>
<body data-bs-theme="<?php echo $varsayilan_tema; ?>">
    <!-- Sidebar (aynı) -->
    <div class="sidebar">
        <div class="sidebar-header">
            <h5 class="mb-0"><i class="bi bi-shield-lock text-primary me-2"></i>Admin Panel</h5>
            <small class="text-muted">MetinPaylaş</small>
        </div>
        <nav class="sidebar-nav">
            <a href="dashboard.php" class="nav-link"><i class="bi bi-speedometer2"></i>Dashboard</a>
            <a href="paylasimlar.php" class="nav-link"><i class="bi bi-file-text"></i>Paylaşımlar</a>
            <a href="kullanicilar.php" class="nav-link active"><i class="bi bi-people"></i>Kullanıcılar</a>
            <a href="ayarlar.php" class="nav-link"><i class="bi bi-gear"></i>Site Ayarları</a>
            <a href="istatistikler.php" class="nav-link"><i class="bi bi-bar-chart"></i>İstatistikler</a>
            <a href="guvenlik.php" class="nav-link"><i class="bi bi-shield-check"></i>Güvenlik</a>
            <hr class="my-3">
            <a href="../index.php" class="nav-link" target="_blank"><i class="bi bi-box-arrow-up-right"></i>Siteyi Görüntüle</a>
            <a href="cikis.php" class="nav-link text-danger"><i class="bi bi-box-arrow-right"></i>Çıkış Yap</a>
        </nav>
    </div>
    
    <!-- Ana İçerik -->
    <div class="main-content">
        <div class="top-bar">
            <div>
                <h4 class="mb-0">Kullanıcılar</h4>
                <small class="text-muted">Sistem kullanıcılarını yönetin</small>
            </div>
            <div>
                <button class="btn btn-outline-secondary btn-sm me-2" id="temaDegistir" title="Tema Değiştir">
                    <i class="bi bi-sun"></i>
                </button>
                <span class="badge bg-primary"><?php echo $_SESSION['admin_yetki']; ?></span>
            </div>
        </div>
        
        <?php if (!empty($mesaj)): ?>
            <div class="alert alert-<?php echo $mesaj_turu; ?> alert-dismissible fade show">
                <?php echo htmlspecialchars($mesaj); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>
        
        <!-- İstatistik Kartları -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="stat-card">
                    <div class="d-flex align-items-center">
                        <div class="stat-icon bg-primary"><i class="bi bi-people"></i></div>
                        <div class="ms-3">
                            <p class="stat-number"><?php echo $toplam_kullanici; ?></p>
                            <p class="stat-label">Toplam Kullanıcı</p>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card">
                    <div class="d-flex align-items-center">
                        <div class="stat-icon bg-success"><i class="bi bi-person-check"></i></div>
                        <div class="ms-3">
                            <p class="stat-number"><?php echo $aktif_kullanici; ?></p>
                            <p class="stat-label">Aktif Kullanıcı</p>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card">
                    <div class="d-flex align-items-center">
                        <div class="stat-icon bg-warning"><i class="bi bi-shield-check"></i></div>
                        <div class="ms-3">
                            <p class="stat-number"><?php echo $admin_sayisi; ?></p>
                            <p class="stat-label">Admin</p>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="text-end">
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addUserModal">
                        <i class="bi bi-person-plus me-1"></i>Yeni Kullanıcı
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Kullanıcılar Tablosu -->
        <div class="card">
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Kullanıcı</th>
                                <th>Email</th>
                                <th>Yetki</th>
                                <th>Durum</th>
                                <th>Kayıt Tarihi</th>
                                <th>Son Giriş</th>
                                <th>İşlemler</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($kullanicilar as $kullanici): ?>
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="avatar bg-primary text-white rounded-circle d-flex align-items-center justify-content-center me-2" 
                                                 style="width: 40px; height: 40px;">
                                                <?php echo strtoupper(substr($kullanici['kullanici_adi'], 0, 1)); ?>
                                            </div>
                                            <div>
                                                <strong><?php echo htmlspecialchars($kullanici['kullanici_adi']); ?></strong>
                                                <?php if (!empty($kullanici['ad_soyad'])): ?>
                                                    <br><small class="text-muted"><?php echo htmlspecialchars($kullanici['ad_soyad']); ?></small>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </td>
                                    <td><?php echo htmlspecialchars($kullanici['email']); ?></td>
                                    <td>
                                        <?php
                                        $yetki_renk = [
                                            'admin' => 'danger',
                                            'moderator' => 'warning',
                                            'kullanici' => 'info'
                                        ];
                                        ?>
                                        <span class="badge bg-<?php echo $yetki_renk[$kullanici['yetki']] ?? 'secondary'; ?>">
                                            <?php echo ucfirst($kullanici['yetki']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($kullanici['durum']): ?>
                                            <span class="badge bg-success">Aktif</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Pasif</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <small><?php echo date('d.m.Y H:i', strtotime($kullanici['kayit_tarihi'])); ?></small>
                                    </td>
                                    <td>
                                        <?php if ($kullanici['son_giris']): ?>
                                            <small><?php echo date('d.m.Y H:i', strtotime($kullanici['son_giris'])); ?></small>
                                        <?php else: ?>
                                            <small class="text-muted">-</small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            <button class="btn btn-outline-primary" 
                                                    onclick="editUser(<?php echo htmlspecialchars(json_encode($kullanici)); ?>)">
                                                <i class="bi bi-pencil"></i>
                                            </button>
                                            <?php if ($kullanici['id'] != $_SESSION['admin_id']): ?>
                                                <button class="btn btn-outline-danger" 
                                                        onclick="deleteUser(<?php echo $kullanici['id']; ?>, '<?php echo htmlspecialchars($kullanici['kullanici_adi']); ?>')">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Kullanıcı Ekleme Modal -->
    <div class="modal fade" id="addUserModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Yeni Kullanıcı Ekle</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="post">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Kullanıcı Adı</label>
                            <input type="text" class="form-control" name="kullanici_adi" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Email</label>
                            <input type="email" class="form-control" name="email" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Ad Soyad</label>
                            <input type="text" class="form-control" name="ad_soyad">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Şifre</label>
                            <input type="password" class="form-control" name="sifre" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Yetki</label>
                            <select class="form-select" name="yetki" required>
                                <option value="kullanici">Kullanıcı</option>
                                <option value="moderator">Moderatör</option>
                                <option value="admin">Admin</option>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                        <button type="submit" name="kullanici_ekle" class="btn btn-primary">Kullanıcı Ekle</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Kullanıcı Düzenleme Modal -->
    <div class="modal fade" id="editUserModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Kullanıcı Düzenle</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="post" id="editUserForm">
                    <input type="hidden" name="kullanici_id" id="edit_id">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Kullanıcı Adı</label>
                            <input type="text" class="form-control" name="kullanici_adi" id="edit_kullanici_adi" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Email</label>
                            <input type="email" class="form-control" name="email" id="edit_email" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Ad Soyad</label>
                            <input type="text" class="form-control" name="ad_soyad" id="edit_ad_soyad">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Yeni Şifre (Boş bırakılırsa değişmez)</label>
                            <input type="password" class="form-control" name="yeni_sifre">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Yetki</label>
                            <select class="form-select" name="yetki" id="edit_yetki" required>
                                <option value="kullanici">Kullanıcı</option>
                                <option value="moderator">Moderatör</option>
                                <option value="admin">Admin</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="durum" id="edit_durum">
                                <label class="form-check-label" for="edit_durum">Aktif</label>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                        <button type="submit" name="kullanici_duzenle" class="btn btn-primary">Güncelle</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Silme Onay Modal -->
    <div class="modal fade" id="deleteModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Kullanıcıyı Sil</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>Bu kullanıcıyı silmek istediğinizden emin misiniz?</p>
                    <div class="alert alert-warning">
                        <strong id="deleteUsername"></strong> kullanıcısı kalıcı olarak silinecek.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <form method="post" style="display: inline;">
                        <input type="hidden" name="kullanici_id" id="deleteUserId">
                        <button type="submit" name="kullanici_sil" class="btn btn-danger">Sil</button>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Tema değiştirme (aynı kod)
        document.addEventListener('DOMContentLoaded', function() {
            const themeToggle = document.getElementById('temaDegistir');
            if (themeToggle) {
                let currentTheme = document.body.getAttribute('data-bs-theme') || 'light';
                updateThemeIcon(currentTheme);
                
                themeToggle.addEventListener('click', function() {
                    const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
                    document.body.setAttribute('data-bs-theme', newTheme);
                    document.documentElement.setAttribute('data-bs-theme', newTheme);
                    localStorage.setItem('theme', newTheme);
                    updateThemeIcon(newTheme);
                    currentTheme = newTheme;
                    
                    document.body.classList.add('theme-transition');
                    setTimeout(() => document.body.classList.remove('theme-transition'), 300);
                    
                    fetch('tema-degistir.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ tema: theme })
                    }).catch(console.error);
                });
                
                function updateThemeIcon(theme) {
                    const icon = themeToggle.querySelector('i');
                    if (icon) icon.className = theme === 'dark' ? 'bi bi-sun' : 'bi bi-moon';
                }
            }
        });
        
        // Kullanıcı düzenleme
        function editUser(user) {
            document.getElementById('edit_id').value = user.id;
            document.getElementById('edit_kullanici_adi').value = user.kullanici_adi;
            document.getElementById('edit_email').value = user.email;
            document.getElementById('edit_ad_soyad').value = user.ad_soyad || '';
            document.getElementById('edit_yetki').value = user.yetki;
            document.getElementById('edit_durum').checked = user.durum == 1;
            
            new bootstrap.Modal(document.getElementById('editUserModal')).show();
        }
        
        // Kullanıcı silme
        function deleteUser(id, username) {
            document.getElementById('deleteUserId').value = id;
            document.getElementById('deleteUsername').textContent = username;
            new bootstrap.Modal(document.getElementById('deleteModal')).show();
        }
    </script>
</body>
</html>

