<?php
/**
 * MetinPaylaş - Admin Paylaşımlar Yönetimi
 */

require_once '../includes/config.php';
require_once '../includes/functions.php';

// Admin girişi kontrolü
if (!isset($_SESSION['admin_giris']) || $_SESSION['admin_giris'] !== true) {
    header('Location: index.php');
    exit;
}

// Ayarları getir
$ayarlar = ayarlariGetir();
$varsayilan_tema = isset($ayarlar['tema']) ? $ayarlar['tema'] : 'light';

$db = baglan();

// Sayfalama
$sayfa = isset($_GET['sayfa']) ? max(1, (int)$_GET['sayfa']) : 1;
$limit = 20;
$offset = ($sayfa - 1) * $limit;

// Arama ve filtreleme
$arama = isset($_GET['arama']) ? trim($_GET['arama']) : '';
$filtre = isset($_GET['filtre']) ? $_GET['filtre'] : 'tumu';

// WHERE koşulları
$where_conditions = [];
$params = [];

if (!empty($arama)) {
    $where_conditions[] = "(baslik LIKE ? OR metin LIKE ? OR id LIKE ?)";
    $params[] = "%$arama%";
    $params[] = "%$arama%";
    $params[] = "%$arama%";
}

switch ($filtre) {
    case 'sifreli':
        $where_conditions[] = "sifre IS NOT NULL";
        break;
    case 'sifresiz':
        $where_conditions[] = "sifre IS NULL";
        break;
    case 'bugun':
        $where_conditions[] = "DATE(tarih) = CURDATE()";
        break;
    case 'bu_hafta':
        $where_conditions[] = "tarih >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
        break;
}

$where_sql = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Toplam kayıt sayısı
$count_query = "SELECT COUNT(*) as toplam FROM paylasimlar $where_sql";
$count_stmt = $db->prepare($count_query);
$count_stmt->execute($params);
$toplam_kayit = $count_stmt->fetch()['toplam'];
$toplam_sayfa = ceil($toplam_kayit / $limit);

// Paylaşımları getir
$query = "SELECT * FROM paylasimlar $where_sql ORDER BY tarih DESC LIMIT $limit OFFSET $offset";
$stmt = $db->prepare($query);
$stmt->execute($params);
$paylasimlar = $stmt->fetchAll();

// Paylaşım silme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['sil_paylasim'])) {
    $silinecek_id = $_POST['paylasim_id'];
    
    // Paylaşımı sil
    $sil_stmt = $db->prepare("DELETE FROM paylasimlar WHERE id = ?");
    if ($sil_stmt->execute([$silinecek_id])) {
        $basari_mesaji = "Paylaşım başarıyla silindi.";
    } else {
        $hata_mesaji = "Paylaşım silinirken bir hata oluştu.";
    }
    
    // Sayfayı yenile
    header("Location: paylasimlar.php?" . http_build_query($_GET));
    exit;
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Paylaşımlar - Admin Panel</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/admin.css">
</head>
<body data-bs-theme="<?php echo $varsayilan_tema; ?>">
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <h5 class="mb-0">
                <i class="bi bi-shield-lock text-primary me-2"></i>
                Admin Panel
            </h5>
            <small class="text-muted">MetinPaylaş</small>
        </div>
        
        <nav class="sidebar-nav">
            <a href="dashboard.php" class="nav-link">
                <i class="bi bi-speedometer2"></i>
                Dashboard
            </a>
            <a href="paylasimlar.php" class="nav-link active">
                <i class="bi bi-file-text"></i>
                Paylaşımlar
            </a>
            <a href="kullanicilar.php" class="nav-link">
                <i class="bi bi-people"></i>
                Kullanıcılar
            </a>
            <a href="ayarlar.php" class="nav-link">
                <i class="bi bi-gear"></i>
                Site Ayarları
            </a>
            <a href="istatistikler.php" class="nav-link">
                <i class="bi bi-bar-chart"></i>
                İstatistikler
            </a>
            <a href="guvenlik.php" class="nav-link">
                <i class="bi bi-shield-check"></i>
                Güvenlik
            </a>
            <hr class="my-3">
            <a href="../index.php" class="nav-link" target="_blank">
                <i class="bi bi-box-arrow-up-right"></i>
                Siteyi Görüntüle
            </a>
            <a href="cikis.php" class="nav-link text-danger">
                <i class="bi bi-box-arrow-right"></i>
                Çıkış Yap
            </a>
        </nav>
    </div>
    
    <!-- Ana İçerik -->
    <div class="main-content">
        <!-- Üst Bar -->
        <div class="top-bar">
            <div>
                <h4 class="mb-0">Paylaşımlar</h4>
                <small class="text-muted">Tüm metin paylaşımlarını yönetin</small>
            </div>
            <div>
                <button class="btn btn-outline-secondary btn-sm me-2" id="temaDegistir" title="Tema Değiştir">
                    <i class="bi bi-sun"></i>
                </button>
                <span class="badge bg-primary"><?php echo $_SESSION['admin_yetki']; ?></span>
            </div>
        </div>
        
        <!-- Filtreler -->
        <div class="card mb-4">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-4">
                        <input type="text" class="form-control" name="arama" 
                               placeholder="Başlık, içerik veya ID ile ara..." 
                               value="<?php echo htmlspecialchars($arama); ?>">
                    </div>
                    <div class="col-md-3">
                        <select class="form-select" name="filtre">
                            <option value="tumu" <?php echo $filtre === 'tumu' ? 'selected' : ''; ?>>Tüm Paylaşımlar</option>
                            <option value="sifreli" <?php echo $filtre === 'sifreli' ? 'selected' : ''; ?>>Şifre Korumalı</option>
                            <option value="sifresiz" <?php echo $filtre === 'sifresiz' ? 'selected' : ''; ?>>Şifresiz</option>
                            <option value="bugun" <?php echo $filtre === 'bugun' ? 'selected' : ''; ?>>Bugün</option>
                            <option value="bu_hafta" <?php echo $filtre === 'bu_hafta' ? 'selected' : ''; ?>>Bu Hafta</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-search me-1"></i>Ara
                        </button>
                    </div>
                    <div class="col-md-3 text-end">
                        <small class="text-muted">
                            Toplam <?php echo number_format($toplam_kayit); ?> paylaşım
                        </small>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- Paylaşımlar Tablosu -->
        <div class="card">
            <div class="card-body p-0">
                <?php if (empty($paylasimlar)): ?>
                    <div class="text-center py-5">
                        <i class="bi bi-file-text text-muted" style="font-size: 3rem;"></i>
                        <h5 class="mt-3 text-muted">Paylaşım bulunamadı</h5>
                        <p class="text-muted">Arama kriterlerinizi değiştirmeyi deneyin.</p>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead class="table-light">
                                <tr>
                                    <th>ID</th>
                                    <th>Başlık</th>
                                    <th>İçerik</th>
                                    <th>Tarih</th>
                                    <th>Görüntülenme</th>
                                    <th>Durum</th>
                                    <th>IP</th>
                                    <th>İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($paylasimlar as $paylasim): ?>
                                    <tr>
                                        <td>
                                            <code><?php echo htmlspecialchars($paylasim['id']); ?></code>
                                        </td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($paylasim['baslik'] ?: 'Başlıksız'); ?></strong>
                                        </td>
                                        <td>
                                            <div style="max-width: 300px;">
                                                <?php 
                                                $icerik = strip_tags($paylasim['metin']);
                                                echo htmlspecialchars(mb_substr($icerik, 0, 100));
                                                if (mb_strlen($icerik) > 100) echo '...';
                                                ?>
                                            </div>
                                            <small class="text-muted">
                                                <?php echo number_format(mb_strlen($icerik)); ?> karakter
                                            </small>
                                        </td>
                                        <td>
                                            <small>
                                                <?php echo date('d.m.Y', strtotime($paylasim['tarih'])); ?><br>
                                                <?php echo date('H:i', strtotime($paylasim['tarih'])); ?>
                                            </small>
                                        </td>
                                        <td>
                                            <span class="badge bg-info">
                                                <?php echo number_format($paylasim['goruntulenme']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if (!empty($paylasim['sifre'])): ?>
                                                <span class="badge bg-warning">
                                                    <i class="bi bi-lock"></i> Şifreli
                                                </span>
                                            <?php else: ?>
                                                <span class="badge bg-success">
                                                    <i class="bi bi-unlock"></i> Açık
                                                </span>
                                            <?php endif; ?>
                                            
                                            <?php if (!empty($paylasim['silinme_suresi'])): ?>
                                                <br><small class="text-warning">
                                                    <i class="bi bi-clock"></i> Otomatik silinecek
                                                </small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <small class="text-muted font-monospace">
                                                <?php echo htmlspecialchars($paylasim['ip_adresi']); ?>
                                            </small>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="../goster.php?id=<?php echo $paylasim['id']; ?>" 
                                                   class="btn btn-outline-primary" target="_blank" title="Görüntüle">
                                                    <i class="bi bi-eye"></i>
                                                </a>
                                                <button type="button" class="btn btn-outline-danger" 
                                                        onclick="confirmDelete('<?php echo $paylasim['id']; ?>', '<?php echo htmlspecialchars($paylasim['baslik'] ?: 'Başlıksız', ENT_QUOTES); ?>')" 
                                                        title="Sil">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Sayfalama -->
        <?php if ($toplam_sayfa > 1): ?>
            <nav class="mt-4">
                <ul class="pagination justify-content-center">
                    <?php if ($sayfa > 1): ?>
                        <li class="page-item">
                            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['sayfa' => $sayfa - 1])); ?>">
                                <i class="bi bi-chevron-left"></i>
                            </a>
                        </li>
                    <?php endif; ?>
                    
                    <?php for ($i = max(1, $sayfa - 2); $i <= min($toplam_sayfa, $sayfa + 2); $i++): ?>
                        <li class="page-item <?php echo $i === $sayfa ? 'active' : ''; ?>">
                            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['sayfa' => $i])); ?>">
                                <?php echo $i; ?>
                            </a>
                        </li>
                    <?php endfor; ?>
                    
                    <?php if ($sayfa < $toplam_sayfa): ?>
                        <li class="page-item">
                            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['sayfa' => $sayfa + 1])); ?>">
                                <i class="bi bi-chevron-right"></i>
                            </a>
                        </li>
                    <?php endif; ?>
                </ul>
            </nav>
        <?php endif; ?>
    </div>
    
    <!-- Silme Onay Modal -->
    <div class="modal fade" id="deleteModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Paylaşımı Sil</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>Bu paylaşımı silmek istediğinizden emin misiniz?</p>
                    <div class="alert alert-warning">
                        <strong id="deleteTitle"></strong> başlıklı paylaşım kalıcı olarak silinecek.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <form method="post" style="display: inline;">
                        <input type="hidden" name="paylasim_id" id="deleteId">
                        <button type="submit" name="sil_paylasim" class="btn btn-danger">
                            <i class="bi bi-trash me-1"></i>Sil
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Admin Tema ve İşlevler -->
    <script>
        // Tema değiştirme (dashboard.php'den kopyalandı)
        document.addEventListener('DOMContentLoaded', function() {
            const themeToggle = document.getElementById('temaDegistir');
            if (themeToggle) {
                let currentTheme = document.body.getAttribute('data-bs-theme') || 'light';
                updateThemeIcon(currentTheme);
                
                themeToggle.addEventListener('click', function() {
                    const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
                    document.body.setAttribute('data-bs-theme', newTheme);
                    document.documentElement.setAttribute('data-bs-theme', newTheme);
                    localStorage.setItem('theme', newTheme);
                    updateThemeIcon(newTheme);
                    currentTheme = newTheme;
                    
                    document.body.classList.add('theme-transition');
                    setTimeout(() => document.body.classList.remove('theme-transition'), 300);
                    
                    updateThemeInDatabase(newTheme);
                });
                
                function updateThemeIcon(theme) {
                    const icon = themeToggle.querySelector('i');
                    if (icon) {
                        icon.className = theme === 'dark' ? 'bi bi-sun' : 'bi bi-moon';
                    }
                }
                
                function updateThemeInDatabase(theme) {
                    fetch('tema-degistir.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ tema: theme })
                    }).then(response => response.json()).then(data => {
                        if (!data.basarili) console.error('Tema güncellenirken hata:', data.mesaj);
                    }).catch(error => console.error('Tema güncelleme hatası:', error));
                }
            }
        });
        
        // Silme onayı
        function confirmDelete(id, title) {
            document.getElementById('deleteId').value = id;
            document.getElementById('deleteTitle').textContent = title;
            new bootstrap.Modal(document.getElementById('deleteModal')).show();
        }
    </script>
</body>
</html>

