/**
 * MetinPaylaş - Gelişmiş JavaScript Dosyası
 */

document.addEventListener('DOMContentLoaded', function() {
    // Tema değiştirme fonksiyonu
    setupThemeToggle();
    
    // Karakter sayacı
    setupCharacterCounter();
    
    // Form doğrulama
    setupFormValidation();
    
    // Animasyonları başlat
    setupAnimations();
});

/**
 * Tema değiştirme işlevselliği
 */
function setupThemeToggle() {
    const themeToggleBtn = document.getElementById('temaDegistir');
    if (!themeToggleBtn) return;
    
    // Önce body data-bs-theme'dan mevcut temayı al (PHP'den gelen)
    const serverTheme = document.body.getAttribute('data-bs-theme') || 'light';
    
    // Local storage'dan tema tercihini kontrol et
    const savedTheme = localStorage.getItem('theme');
    const finalTheme = savedTheme || serverTheme;
    
    // Temayı uygula
    document.documentElement.setAttribute('data-bs-theme', finalTheme);
    document.body.setAttribute('data-bs-theme', finalTheme);
    localStorage.setItem('theme', finalTheme);
    updateThemeIcon(finalTheme);
    
    // Eğer localStorage'daki tema sunucudakinden farklıysa, sunucuyu güncelle
    if (savedTheme && savedTheme !== serverTheme) {
        updateThemeInDatabase(savedTheme);
    }
    
    // Tema değiştirme butonu tıklama olayı
    themeToggleBtn.addEventListener('click', function() {
        const currentTheme = document.documentElement.getAttribute('data-bs-theme');
        const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
        
        // Temayı görsel olarak değiştir
        document.documentElement.setAttribute('data-bs-theme', newTheme);
        document.body.setAttribute('data-bs-theme', newTheme);
        localStorage.setItem('theme', newTheme);
        updateThemeIcon(newTheme);
        
        // Tema değiştiğinde animasyon ekle
        document.body.classList.add('theme-transition');
        setTimeout(() => {
            document.body.classList.remove('theme-transition');
        }, 1000);
        
        // Temayı veritabanında güncelle
        updateThemeInDatabase(newTheme);
    });
}

/**
 * Temayı veritabanında güncelle
 */
function updateThemeInDatabase(theme) {
    fetch('tema-degistir.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ tema: theme })
    })
    .then(response => response.json())
    .then(data => {
        if (data.basarili) {
            console.log('Tema veritabanında güncellendi:', theme);
        } else {
            console.error('Tema güncellenirken hata:', data.mesaj);
        }
    })
    .catch(error => {
        console.error('Tema güncelleme hatası:', error);
    });
}

/**
 * Tema ikonunu günceller
 * 
 * @param {string} theme Tema adı ('dark' veya 'light')
 */
function updateThemeIcon(theme) {
    const themeToggleBtn = document.getElementById('temaDegistir');
    if (!themeToggleBtn) return;
    
    const iconElement = themeToggleBtn.querySelector('i') || document.createElement('i');
    iconElement.className = theme === 'dark' ? 'bi bi-sun' : 'bi bi-moon-stars';
    
    if (!themeToggleBtn.querySelector('i')) {
        themeToggleBtn.prepend(iconElement);
    }
}

/**
 * Karakter sayacı işlevselliği
 */
function setupCharacterCounter() {
    const textarea = document.getElementById('metin');
    const counter = document.getElementById('karakterSayaci');
    
    if (!textarea || !counter) return;
    
    // İlk yüklemede sayacı güncelle
    counter.textContent = textarea.value.length;
    
    // Her değişiklikte sayacı güncelle
    textarea.addEventListener('input', function() {
        counter.textContent = this.value.length;
        
        // Karakter sayısına göre renk değiştir
        if (this.value.length > 10000) {
            counter.classList.add('text-warning');
            counter.classList.remove('text-success', 'text-danger');
        } else if (this.value.length > 50000) {
            counter.classList.add('text-danger');
            counter.classList.remove('text-success', 'text-warning');
        } else if (this.value.length > 0) {
            counter.classList.add('text-success');
            counter.classList.remove('text-warning', 'text-danger');
        } else {
            counter.classList.remove('text-success', 'text-warning', 'text-danger');
        }
    });
}

/**
 * Form doğrulama işlevselliği
 */
function setupFormValidation() {
    // Sadece metin paylaşım formunu hedefle
    const form = document.querySelector('form[method="post"][enctype="multipart/form-data"]');
    if (!form) return;
    
    form.addEventListener('submit', function(event) {
        const textarea = document.getElementById('metin');
        
        // Metin alanı boş mu kontrol et
        if (!textarea || textarea.value.trim().length === 0) {
            event.preventDefault();
            showToast('Lütfen paylaşmak istediğiniz metni girin.', 'danger');
            if (textarea) textarea.focus();
            return false;
        }
        
        // Çok uzun metin kontrolü (1MB'dan büyük metinleri engelle)
        if (textarea.value.length > 1000000) {
            event.preventDefault();
            showToast('Metin çok uzun. Lütfen 1MB\'dan küçük bir metin girin.', 'danger');
            return false;
        }
        
        // Form gönderilmeden önce buton durumunu güncelle
        const submitButton = form.querySelector('button[type="submit"]');
        if (submitButton) {
            submitButton.disabled = true;
            submitButton.innerHTML = '<span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span> İşleniyor...';
        }
        
        return true;
    });
    
    // Markdown desteği checkbox'ı varsa
    const markdownCheckbox = document.getElementById('markdownDestegi');
    if (markdownCheckbox) {
        markdownCheckbox.addEventListener('change', function() {
            const textarea = document.getElementById('metin');
            if (!textarea) return;
            
            if (this.checked) {
                textarea.classList.add('markdown-editor');
            } else {
                textarea.classList.remove('markdown-editor');
            }
        });
    }
}

/**
 * Animasyonları başlatma
 */
function setupAnimations() {
    // Sayfa yüklendiğinde header animasyonu
    const header = document.querySelector('header');
    if (header) {
        header.classList.add('animate-header');
    }
    
    // Form alanları için giriş animasyonu
    const formElements = document.querySelectorAll('.form-control, .form-select, .btn-lg');
    formElements.forEach((element, index) => {
        setTimeout(() => {
            element.classList.add('animate-form-element');
        }, 100 * index);
    });
}

/**
 * Toast bildirimi gösterme fonksiyonu
 * Bu fonksiyon index.php ve goster.php içinde ayrıca tanımlanmıştır
 * 
 * @param {string} message Gösterilecek mesaj
 * @param {string} type Bildirim tipi (success, danger, warning, info)
 */
function showToast(message, type = 'success') {
    // Toast elementi oluştur
    const toastEl = document.createElement('div');
    toastEl.className = `toast align-items-center text-white bg-${type} border-0`;
    toastEl.setAttribute('role', 'alert');
    toastEl.setAttribute('aria-live', 'assertive');
    toastEl.setAttribute('aria-atomic', 'true');
    
    const flexBox = document.createElement('div');
    flexBox.className = 'd-flex';
    
    const toastBody = document.createElement('div');
    toastBody.className = 'toast-body d-flex align-items-center';
    
    // İkon ekle
    const icon = document.createElement('i');
    icon.className = type === 'success' ? 'bi bi-check-circle-fill me-2' : 'bi bi-exclamation-triangle-fill me-2';
    toastBody.appendChild(icon);
    
    // Mesaj ekle
    toastBody.appendChild(document.createTextNode(message));
    
    // Kapat butonu ekle
    const closeBtn = document.createElement('button');
    closeBtn.type = 'button';
    closeBtn.className = 'btn-close btn-close-white ms-auto me-2';
    closeBtn.setAttribute('data-bs-dismiss', 'toast');
    closeBtn.setAttribute('aria-label', 'Kapat');
    
    flexBox.appendChild(toastBody);
    flexBox.appendChild(closeBtn);
    toastEl.appendChild(flexBox);
    
    // Toast konteyner oluştur veya mevcut olanı bul
    let toastContainer = document.querySelector('.toast-container');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.className = 'toast-container position-fixed bottom-0 end-0 p-3';
        document.body.appendChild(toastContainer);
    }
    
    // Çok fazla toast mesajı varsa eski olanları temizle
    const existingToasts = toastContainer.querySelectorAll('.toast');
    if (existingToasts.length > 3) {
        existingToasts.forEach((toast, index) => {
            if (index < existingToasts.length - 3) {
                toast.remove();
            }
        });
    }
    
    // Toast'u konteynere ekle
    toastContainer.appendChild(toastEl);
    
    // Bootstrap Toast nesnesini oluştur ve göster
    const toast = new bootstrap.Toast(toastEl, {
        animation: true,
        autohide: true,
        delay: 3000
    });
    toast.show();
    
    // Kapatıldıktan sonra DOM'dan kaldır
    toastEl.addEventListener('hidden.bs.toast', function () {
        toastEl.remove();
    });
}