<?php
/**
 * MetinPaylaş - Yardımcı Fonksiyonlar
 * 
 * Bu dosya sistemde kullanılan yardımcı fonksiyonları içerir
 */

// Yapılandırma dosyasını dahil et
require_once 'config.php';

/**
 * Yeni bir metin paylaşımı oluşturur
 * 
 * @param string $metin Paylaşılacak metin içeriği
 * @param string|null $baslik Başlık (isteğe bağlı)
 * @param string|null $sifre Şifre (isteğe bağlı)
 * @param int|null $silinme_suresi Otomatik silinme süresi (dakika cinsinden)
 * @param string|null $ozel_url Özel URL (isteğe bağlı)
 * @param bool $goruntu_sayaci_goster Görüntüleme sayacını göster
 * @param bool $markdown_kullan Markdown formatını kullan
 * @return array İşlem sonucu ve oluşturulan ID
 */
function metinOlustur($metin, $baslik = null, $sifre = null, $silinme_suresi = null, $ozel_url = null, $goruntu_sayaci_goster = true, $markdown_kullan = false) {
    // HTML içeriği güvenli hale getirme işlemi yapma (CKEditor kullanıldığı için HTML içeriği korunmalı)
    // $metin = htmlspecialchars($metin, ENT_QUOTES, 'UTF-8'); // Bu satırı kaldırıyoruz
    $db = baglan();
    
    // Boş metin kontrolü
    if (empty(trim($metin))) {
        return ['basarili' => false, 'mesaj' => 'Lütfen paylaşmak istediğiniz metni girin.'];
    }
    
    // Şifre varsa hash'le
    if (!empty($sifre)) {
        $sifre = password_hash($sifre, PASSWORD_DEFAULT);
    } else {
        $sifre = null;
    }
    
    // Özel URL kontrolü
    if (!empty($ozel_url)) {
        // Özel URL'yi güvenli hale getir
        $ozel_url = preg_replace('/[^a-zA-Z0-9-_]/', '', $ozel_url);
        
        // Özel URL'nin benzersiz olduğunu kontrol et
        $url_kontrol = $db->prepare("SELECT id FROM paylasimlar WHERE ozel_url = ?");
        $url_kontrol->execute([$ozel_url]);
        
        if ($url_kontrol->rowCount() > 0) {
            return ['basarili' => false, 'mesaj' => 'Bu özel URL zaten kullanılıyor. Lütfen başka bir URL deneyin.'];
        }
    }
    
    // Benzersiz ID oluştur
    $id = benzersizID();
    $sorgu = $db->prepare("SELECT id FROM paylasimlar WHERE id = ?");
    
    // ID'nin benzersiz olduğundan emin ol
    do {
        $sorgu->execute([$id]);
        if ($sorgu->rowCount() > 0) {
            $id = benzersizID();
        }
    } while ($sorgu->rowCount() > 0);
    
    // Metni veritabanına kaydet
    try {
        $sorgu = $db->prepare("INSERT INTO paylasimlar (id, baslik, metin, sifre, silinme_suresi, ip_adresi, ozel_url, goruntu_sayaci_goster, markdown_kullan) 
                              VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $sonuc = $sorgu->execute([
            $id, 
            $baslik, 
            $metin, 
            $sifre, 
            $silinme_suresi, 
            kullaniciIP(), 
            $ozel_url, 
            $goruntu_sayaci_goster ? 1 : 0, 
            $markdown_kullan ? 1 : 0
        ]);
        
        if ($sonuc) {
            return ['basarili' => true, 'id' => $id, 'ozel_url' => $ozel_url];
        } else {
            return ['basarili' => false, 'mesaj' => 'Metin kaydedilirken bir hata oluştu.'];
        }
    } catch (PDOException $e) {
        return ['basarili' => false, 'mesaj' => 'Veritabanı hatası: ' . $e->getMessage()];
    }
}

/**
 * ID'ye göre metin paylaşımını getirir
 * 
 * @param string $id Paylaşım ID'si veya özel URL
 * @return array|false Paylaşım bilgileri veya false
 */
function metinGetir($id) {
    $db = baglan();
    
    try {
        // Önce ID'ye göre ara
        $sorgu = $db->prepare("SELECT * FROM paylasimlar WHERE id = ? OR ozel_url = ?");
        $sorgu->execute([$id, $id]);
        
        if ($sorgu->rowCount() > 0) {
            $paylasim = $sorgu->fetch();
            
            // Görüntülenme sayısını artır
            $guncelle = $db->prepare("UPDATE paylasimlar SET goruntulenme = goruntulenme + 1 WHERE id = ?");
            $guncelle->execute([$paylasim['id']]);
            
            // Silinme süresi kontrolü
            if (!empty($paylasim['silinme_suresi'])) {
                $olusturulma = new DateTime($paylasim['tarih']);
                $silinme = clone $olusturulma;
                $silinme->add(new DateInterval('PT' . $paylasim['silinme_suresi'] . 'M'));
                $simdi = new DateTime();
                
                if ($simdi > $silinme) {
                    // Süresi dolmuş, sil
                    $sil = $db->prepare("DELETE FROM paylasimlar WHERE id = ?");
                    $sil->execute([$paylasim['id']]);
                    return false;
                }
            }
            
            // Paylaşıma ait resimleri getir
            $resimler_sorgu = $db->prepare("SELECT * FROM resimler WHERE paylasim_id = ? ORDER BY yuklenme_tarihi");
            $resimler_sorgu->execute([$paylasim['id']]);
            $paylasim['resimler'] = $resimler_sorgu->fetchAll();
            
            return $paylasim;
        } else {
            return false;
        }
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Şifre korumalı paylaşımın şifresini kontrol eder
 * 
 * @param string $id Paylaşım ID'si
 * @param string $sifre Kontrol edilecek şifre
 * @return bool Şifre doğru mu
 */
function sifreKontrol($id, $sifre) {
    $db = baglan();
    
    $sorgu = $db->prepare("SELECT sifre FROM paylasimlar WHERE id = ? OR ozel_url = ?");
    $sorgu->execute([$id, $id]);
    
    if ($sorgu->rowCount() > 0) {
        $paylasim = $sorgu->fetch();
        
        if (!empty($paylasim['sifre'])) {
            return password_verify($sifre, $paylasim['sifre']);
        }
    }
    
    return false;
}

/**
 * Markdown formatını HTML'e dönüştürür
 * 
 * @param string $metin Markdown formatında metin
 * @return string HTML formatında metin
 */
function markdownToHtml($metin) {
    // Basit markdown dönüşümleri
    $metin = preg_replace('/\*\*(.*?)\*\*/s', '<strong>$1</strong>', $metin);
    $metin = preg_replace('/\*(.*?)\*/s', '<em>$1</em>', $metin);
    $metin = preg_replace('/\~\~(.*?)\~\~/s', '<del>$1</del>', $metin);
    $metin = preg_replace('/\`(.*?)\`/s', '<code>$1</code>', $metin);
    
    // Başlıklar
    $metin = preg_replace('/^# (.*?)$/m', '<h1>$1</h1>', $metin);
    $metin = preg_replace('/^## (.*?)$/m', '<h2>$1</h2>', $metin);
    $metin = preg_replace('/^### (.*?)$/m', '<h3>$1</h3>', $metin);
    
    // Listeler
    $metin = preg_replace('/^- (.*?)$/m', '<li>$1</li>', $metin);
    $metin = preg_replace('/^([0-9]+)\. (.*?)$/m', '<li>$2</li>', $metin);
    $metin = preg_replace('/((?:<li>.*?<\/li>[\n]*)+)/', '<ul>$1</ul>', $metin);
    
    // Linkler
    $metin = preg_replace('/\[(.*?)\]\((.*?)\)/s', '<a href="$2" target="_blank">$1</a>', $metin);
    
    // Resimler
    $metin = preg_replace('/!\[(.*?)\]\((.*?)\)/s', '<img src="$2" alt="$1" class="img-fluid">', $metin);
    
    // Kod blokları
    $metin = preg_replace('/```(.*?)```/s', '<pre><code>$1</code></pre>', $metin);
    
    // Alıntılar
    $metin = preg_replace('/^> (.*?)$/m', '<blockquote>$1</blockquote>', $metin);
    
    // Yatay çizgi
    $metin = preg_replace('/^---$/m', '<hr>', $metin);
    
    // Satır sonları
    $metin = nl2br($metin);
    
    return $metin;
}

/**
 * Resim yükler ve veritabanına kaydeder
 * 
 * @param string $paylasim_id Paylaşım ID'si
 * @param array $dosya $_FILES dizisinden dosya bilgisi
 * @return array İşlem sonucu
 */
function resimYukle($paylasim_id, $dosya) {
    $db = baglan();
    
    // Paylaşımın var olduğunu kontrol et
    $paylasim_kontrol = $db->prepare("SELECT id FROM paylasimlar WHERE id = ?");
    $paylasim_kontrol->execute([$paylasim_id]);
    
    if ($paylasim_kontrol->rowCount() == 0) {
        return ['basarili' => false, 'mesaj' => 'Geçersiz paylaşım ID\'si.'];
    }
    
    // Ayarlardan izin verilen uzantıları ve maksimum dosya boyutunu al
    $ayarlar = ayarlariGetir(['izin_verilen_uzantilar', 'max_dosya_boyutu']);
    $izin_verilen_uzantilar = explode(',', $ayarlar['izin_verilen_uzantilar']);
    $max_dosya_boyutu = (int)$ayarlar['max_dosya_boyutu'];
    
    // Dosya uzantısını kontrol et
    $dosya_adi = $dosya['name'];
    $dosya_uzantisi = strtolower(pathinfo($dosya_adi, PATHINFO_EXTENSION));
    
    if (!in_array($dosya_uzantisi, $izin_verilen_uzantilar)) {
        return ['basarili' => false, 'mesaj' => 'Bu dosya türüne izin verilmiyor. İzin verilen uzantılar: ' . implode(', ', $izin_verilen_uzantilar)];
    }
    
    // Dosya boyutunu kontrol et
    if ($dosya['size'] > $max_dosya_boyutu) {
        return ['basarili' => false, 'mesaj' => 'Dosya boyutu çok büyük. Maksimum boyut: ' . formatBytes($max_dosya_boyutu)];
    }
    
    // Benzersiz dosya adı oluştur
    $yeni_dosya_adi = uniqid() . '_' . temizDosyaAdi($dosya_adi);
    $yuklenecek_dizin = 'uploads/';
    
    // Uploads dizini yoksa oluştur
    if (!file_exists($yuklenecek_dizin)) {
        mkdir($yuklenecek_dizin, 0755, true);
    }
    
    $hedef_yol = $yuklenecek_dizin . $yeni_dosya_adi;
    
    // Dosyayı yükle
    if (move_uploaded_file($dosya['tmp_name'], $hedef_yol)) {
        // Veritabanına kaydet
        try {
            $sorgu = $db->prepare("INSERT INTO resimler (paylasim_id, dosya_adi, dosya_yolu, boyut) VALUES (?, ?, ?, ?)");
            $sonuc = $sorgu->execute([$paylasim_id, $dosya_adi, $hedef_yol, $dosya['size']]);
            
            if ($sonuc) {
                return [
                    'basarili' => true, 
                    'mesaj' => 'Dosya başarıyla yüklendi.',
                    'dosya_adi' => $dosya_adi,
                    'dosya_yolu' => $hedef_yol,
                    'id' => $db->lastInsertId()
                ];
            } else {
                // Dosya yüklendi ama veritabanına kaydedilemedi, dosyayı sil
                unlink($hedef_yol);
                return ['basarili' => false, 'mesaj' => 'Dosya yüklendi ancak veritabanına kaydedilemedi.'];
            }
        } catch (PDOException $e) {
            // Hata durumunda dosyayı sil
            unlink($hedef_yol);
            return ['basarili' => false, 'mesaj' => 'Veritabanı hatası: ' . $e->getMessage()];
        }
    } else {
        return ['basarili' => false, 'mesaj' => 'Dosya yüklenirken bir hata oluştu.'];
    }
}

/**
 * Dosya adını temizler (özel karakterleri kaldırır)
 * 
 * @param string $dosya_adi Temizlenecek dosya adı
 * @return string Temizlenmiş dosya adı
 */
function temizDosyaAdi($dosya_adi) {
    // Türkçe karakterleri değiştir
    $turkce = ['ı', 'ğ', 'ü', 'ş', 'ö', 'ç', 'İ', 'Ğ', 'Ü', 'Ş', 'Ö', 'Ç'];
    $ingilizce = ['i', 'g', 'u', 's', 'o', 'c', 'I', 'G', 'U', 'S', 'O', 'C'];
    $dosya_adi = str_replace($turkce, $ingilizce, $dosya_adi);
    
    // Sadece alfanumerik karakterler, nokta, tire ve alt çizgi kalacak şekilde temizle
    $dosya_adi = preg_replace('/[^a-zA-Z0-9._-]/', '', $dosya_adi);
    
    return $dosya_adi;
}

/**
 * Byte değerini okunabilir formata dönüştürür
 * 
 * @param int $bytes Byte değeri
 * @return string Okunabilir boyut (örn. 5.2 MB)
 */
function formatBytes($bytes) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    
    $bytes /= pow(1024, $pow);
    
    return round($bytes, 2) . ' ' . $units[$pow];
}

/**
 * Ayarları veritabanından getirir
 * 
 * @param array|null $anahtarlar İstenen ayar anahtarları (null ise tüm ayarları getirir)
 * @return array Ayarlar
 */
function ayarlariGetir($anahtarlar = null) {
    $db = baglan();
    $ayarlar = [];
    
    try {
        if ($anahtarlar === null) {
            // Tüm ayarları getir
            $sorgu = $db->query("SELECT anahtar, deger FROM ayarlar");
        } else {
            // Belirli ayarları getir
            $in = str_repeat('?,', count($anahtarlar) - 1) . '?';
            $sorgu = $db->prepare("SELECT anahtar, deger FROM ayarlar WHERE anahtar IN ($in)");
            $sorgu->execute($anahtarlar);
        }
        
        while ($ayar = $sorgu->fetch()) {
            $ayarlar[$ayar['anahtar']] = $ayar['deger'];
        }
        
        return $ayarlar;
    } catch (PDOException $e) {
        // Hata durumunda varsayılan ayarları döndür
        return [
            'site_baslik' => 'MetinPaylaş',
            'site_aciklama' => 'Hızlı ve kolay metin paylaşım platformu',
            'max_dosya_boyutu' => 5242880, // 5MB
            'izin_verilen_uzantilar' => 'jpg,jpeg,png,gif',
            'kayit_acik' => 1,
            'tema' => 'light',
            'markdown_varsayilan' => 1
        ];
    }
}

/**
 * Süresi dolan paylaşımları temizler
 * (Cron job ile çalıştırılmalı)
 */
function suresiDolanTemizle() {
    $db = baglan();
    
    try {
        $sorgu = $db->prepare("SELECT id, tarih, silinme_suresi FROM paylasimlar WHERE silinme_suresi IS NOT NULL");
        $sorgu->execute();
        
        while ($paylasim = $sorgu->fetch()) {
            $olusturulma = new DateTime($paylasim['tarih']);
            $silinme = clone $olusturulma;
            $silinme->add(new DateInterval('PT' . $paylasim['silinme_suresi'] . 'M'));
            $simdi = new DateTime();
            
            if ($simdi > $silinme) {
                // Paylaşıma ait resimleri sil
                $resimler_sorgu = $db->prepare("SELECT dosya_yolu FROM resimler WHERE paylasim_id = ?");
                $resimler_sorgu->execute([$paylasim['id']]);
                
                while ($resim = $resimler_sorgu->fetch()) {
                    if (file_exists($resim['dosya_yolu'])) {
                        unlink($resim['dosya_yolu']);
                    }
                }
                
                // Paylaşımı sil
                $sil = $db->prepare("DELETE FROM paylasimlar WHERE id = ?");
                $sil->execute([$paylasim['id']]);
            }
        }
        
        return true;
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Ayar güncelle
 */
function ayarGuncelle($anahtar, $deger) {
    $db = baglan();
    
    try {
        // Önce ayarın mevcut olup olmadığını kontrol et
        $kontrol = $db->prepare("SELECT id FROM ayarlar WHERE anahtar = ?");
        $kontrol->execute([$anahtar]);
        
        if ($kontrol->rowCount() > 0) {
            // Mevcut ayarı güncelle
            $sorgu = $db->prepare("UPDATE ayarlar SET deger = ? WHERE anahtar = ?");
            $sonuc = $sorgu->execute([$deger, $anahtar]);
        } else {
            // Yeni ayar ekle
            $sorgu = $db->prepare("INSERT INTO ayarlar (anahtar, deger) VALUES (?, ?)");
            $sonuc = $sorgu->execute([$anahtar, $deger]);
        }
        
        return $sonuc;
    } catch (PDOException $e) {
        error_log("Ayar güncelleme hatası: " . $e->getMessage());
        return false;
    }
}
?>